//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Collections;
using System.Reflection;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents the base class from which all implementations of <see cref="IDictionary"/> parameter attributes must derive.
	/// </summary>
    public class DictionaryConverterAttribute : CollectionConverterBaseAttribute
	{
		private readonly Type dictionaryType;
		private readonly string keyPropertyName;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="dictionaryType"></param>
        /// <param name="keyPropertyName"></param>
        public DictionaryConverterAttribute(Type dictionaryType, string keyPropertyName)
            : this(dictionaryType, keyPropertyName, null)
        {
        }

		/// <summary>
		/// Initialize a new instance of the <see cref="DictionaryPropertyParameterAttribute"/> class with the order, the parameter type, the property name, the type of dictionary to create and the property name of the nested element that is the key for the dictionary.
		/// </summary>
		/// <param name="order">The ordinal position of the constructor parameter.</param>
		/// <param name="parameterType">The parameter type.</param>
		/// <param name="propertyName">The name of the property to get the parameter value.</param>
		/// <param name="dictionaryType">The type of dictionary to create.</param>
		/// <param name="keyPropertyName">The property of the nested element that is the key for the dictionary.</param>
        public DictionaryConverterAttribute(Type dictionaryType, string keyPropertyName, Type elementFactoryType)
            : base(elementFactoryType)
		{
			this.dictionaryType = dictionaryType;
			this.keyPropertyName = keyPropertyName;
		}

		/// <summary>
		/// Gets the type of dictionary to create.
		/// </summary>
		/// <value>
		/// The type of dictionary to create.
		/// </value>
		public Type DictionaryType
		{
			get { return dictionaryType; }
		}

		/// <summary>
		/// Gets the property name of the nested element that will be the key for the dictioanry.
		/// </summary>
		/// <value>
		/// The property name of the nested element that will be the key for the dictioanry.
		/// </value>
		public string KeyPropertyName
		{
			get { return keyPropertyName; }
		}

		/// <summary>
		/// Gets the <see cref="DictionaryFactory"/> to use to transform the property value to the value for the parameter.
		/// </summary>
        /// <returns>The <see cref="DictionaryFactory"/> to use to transform the property value to the value for the parameter.</returns>
        protected internal override IObjectFactory<TObjectToCreate, TObjectConfiguration> GetFactory<TObjectToCreate, TObjectConfiguration>(TObjectConfiguration objectConfiguration)
		{
            Type elementType = typeof(TObjectToCreate).GetGenericArguments()[0];
            Type elementConfigurationType = typeof(TObjectConfiguration).GetGenericArguments()[0];
            Type elementKey = elementConfigurationType.GetProperty(keyPropertyName).PropertyType;

            Type dictionaryFactory = typeof(DictionaryFactory<,,>);
            dictionaryFactory = dictionaryFactory.MakeGenericType(elementKey, elementType, elementConfigurationType);

            object elementFactory = GetElementFactoryGeneric(elementType, elementConfigurationType);

            return (IObjectFactory<TObjectToCreate, TObjectConfiguration>)Activator.CreateInstance(dictionaryFactory, dictionaryFactory, elementFactory);
		}

    }
}
